# Данный пример выводит аппроксимацию мендра, с помощью ряда фурье на 1 и 2 канал ЦАП

from lpciePy import X502, DacOutFlags, Streams, SyncMode

from common import find_devices, select_device
import math

BASE_FREQ = 2000 # Частота меандра
DAC_FREQ = 500000 # Частота вывода ЦАП

def configure(device: X502):
    device.SetSyncMode(SyncMode.X502_SYNC_INTERNAL) # Устанавливаем внутреннюю синхронизацию
    device.StreamsStop() # Останавливаем все потоки данных
    device.StreamsDisable(Streams.X502_STREAM_ALL_IN | Streams.X502_STREAM_ALL_OUT) # Запрещаем все потоки

    outFreq = device.SetOutFreq(DAC_FREQ) # Устанавливаем частоту вывода ЦАП
    device.StreamsEnable(Streams.X502_STREAM_DAC1 | Streams.X502_STREAM_DAC2) # Разрешаем поток для первого и второго каналов ЦАП
    device.Configure() # Записываем настройки в модуль

    return outFreq


def output_sin(device: X502, outFreq):

    AMP = 8 # Амплитуда сигнала

    device.StreamsStart() # Запускаем разрешенные потоки

    partial_sum_n = 10

    # Создаем массивы для генерации синусов
    phases1 = [0 for _ in range(1, partial_sum_n, 2)]
    deltas1 = [2 * math.pi * (BASE_FREQ * (i)) / outFreq for i in range(1, partial_sum_n, 2)]
    amplitudes1 = [AMP / (i) for i in range(1, partial_sum_n, 2)]

    phases2 = [0 for _ in range(0, partial_sum_n, 2)]
    deltas2 = [2 * math.pi * (BASE_FREQ * (i+1)) / outFreq for i in range(0, partial_sum_n, 2)]
    amplitudes2 = [AMP / (i+1) for i in range(0, partial_sum_n, 2)]

    while True:

        block_size = int(outFreq)

        dac1 = [0.0] * block_size
        dac2 = [0.0] * block_size

        for i in range(block_size):
            val1, val2 = 0, 0
            for phase1, amp1, phase2, amp2 in zip(phases1, amplitudes1, phases2, amplitudes2):
                # Суммируем синусы
                val1 += amp1 * math.sin(phase1)
                val2 += amp2 * math.sin(phase2)
            # Добавляем значения в массив
            dac1[i] = val1
            dac2[i] = val2

            # Сдвигаем фазу
            phases1 = [phase + delta for phase, delta in zip(phases1, deltas1)]
            phases2 = [phase + delta for phase, delta in zip(phases2, deltas2)]

        # Подготавливаем данные к отправке
        data = device.PrepareData(dac1, dac2, None, DacOutFlags.X502_DAC_FLAGS_VOLT)

        # Функция может отправить не все данные, проверяем сколько отсчетов было отправлено,
        # и досылаем, если были отправлены не все
        sent = 0
        while sent < outFreq:
            sent += device.Send(data[sent:])


def main():

    # ВАЖНО!! Необходимо оставлять переменные которые содержат записи об устройствах "живыми" до вызова OpenByDevRecord().
    # Иначе, вызовы вида  select_device(find_devices()) и X502.OpenByDevRecord(select_device(list)) приведут к ошибке сегментации,
    # так как временный объект, который создается при вложенных вызовах, будет освобожден раньше времени, что приведёт к ошибке use-after-free

    list = find_devices() # Получаем список всех подключенных устройств
    record = select_device(list) # Выбираем устройство
    device = X502.OpenByDevRecord(record) # Открываем выбранное устройство

    outFreq = configure(device)

    output_sin(device, int(outFreq))

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print()
        exit(0)