# Данный пример выводит синус на 1 канал ЦАП, в режиме циклического буфера

from lpciePy import X502, DacOutFlags, Streams, OutCycleFlags, SyncMode

from common import find_devices, select_device
import math

SIN_FREQ = 12000 # Частота генерируемого синуса
DAC_FREQ = 500000 # Частота вывода ЦАП

def configure(device: X502):
    device.SetSyncMode(SyncMode.X502_SYNC_INTERNAL) # Устанавливаем внутреннюю синхронизацию
    device.StreamsStop() # Останавливаем все потоки данных
    device.StreamsDisable(Streams.X502_STREAM_ALL_IN | Streams.X502_STREAM_ALL_OUT) # Запрещаем все потоки

    outFreq = device.SetOutFreq(DAC_FREQ) # Устанавливаем частоту вывода ЦАП
    device.StreamsEnable(Streams.X502_STREAM_DAC1) # Разрешаем поток данных для первого канала ЦАП
    device.Configure() # Записываем настройки в модуль

    return outFreq


def output_sin(device: X502, outFreq):
    AMP = 4 # Амплитуда сигнала

    buf_size = int(outFreq / SIN_FREQ) # Вычисляем размер буфера для одного периода синуса
    device.OutCycleLoadStart(buf_size) # Устанавливаем рамер циклического буфера

    phase = 0
    delta = 2 * math.pi * SIN_FREQ / outFreq

    dac = [0.0] * buf_size

    # Вычисляем сигнал
    for i in range(buf_size):
        dac[i] = AMP * math.sin(phase)
        phase += delta

    # Подготавливаем данные к отправке
    data = device.PrepareData(dac, None, None, DacOutFlags.X502_DAC_FLAGS_VOLT)

    # Функция может отправить не все данные, проверяем сколько отсчетов было отправлено,
    # и досылаем, если были отправлены не все
    sent = 0
    while sent < outFreq:
        sent += device.Send(data[sent:])

    # Делаем активным загруженный сигнал
    device.OutCycleSetup(OutCycleFlags.X502_OUT_CYCLE_FLAGS_WAIT_DONE)
    device.StreamsStart() # Запускаем вывод

def main():

    # ВАЖНО!! Необходимо оставлять переменные которые содержат записи об устройствах "живыми" до вызова OpenByDevRecord().
    # Иначе, вызовы вида  select_device(find_devices()) и X502.OpenByDevRecord(select_device(list)) приведут к ошибке сегментации,
    # так как временный объект, который создается при вложенных вызовах, будет освобожден раньше времени, что приведёт к ошибке use-after-free

    list = find_devices() # Получаем список всех подключенных устройств
    record = select_device(list) # Выбираем устройство
    device = X502.OpenByDevRecord(record) # Открываем выбранное устройство

    outFreq = configure(device)

    output_sin(device, int(outFreq))
    while True:
        pass

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print()
        exit(0)